<?php

/**
 * Template Name: LoginPress
 *
 * Template to display the WordPress login page in the WP-Customizer.
 *
 * @package LoginPress
 * @author  WPBrigade
 * @since 1.1.3
 */

// Redirect if viewed from outside the Customizer.
global $interim_login;

if ( ! is_customize_preview() ) {

	$loginpress_obj  = new LoginPress();
	$loginpress_page = $loginpress_obj->get_loginpress_page();

	$page = get_permalink( $loginpress_page );

	// Generate the redirect url.
	$url = add_query_arg(
		array(
			'autofocus[panel]' => 'loginpress_panel',
			'return'           => admin_url( 'index.php' ),
			'url'              => rawurlencode( $page ),
		),
		admin_url( 'customize.php' )
	);

	wp_safe_redirect( $url );
}

	/** Make sure that the WordPress bootstrap has run before continuing. */
	require( ABSPATH . '/wp-load.php' );

	// Redirect to https login if forced to use SSL
	if ( force_ssl_admin() && ! is_ssl() ) {
		if ( 0 === strpos($_SERVER['REQUEST_URI'], 'http') ) {
			wp_redirect( set_url_scheme( $_SERVER['REQUEST_URI'], 'https' ) );
			exit();
		} else {
			wp_redirect( 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
			exit();
		}
	}

/**
* Output the login page header.
*
* @param string   $title    Optional. WordPress login Page title to display in the `<title>` element.
*                           Default 'Log In'.
* @param string   $message  Optional. Message to display in header. Default empty.
* @param WP_Error $wp_error Optional. The error to pass. Default empty.
*/
function login_header( $title = 'Log In', $message = '', $wp_error = '' ) {
	global $error, $interim_login, $action;

	// Don't index any of these forms
	add_action( 'login_head', 'wp_no_robots' );

	add_action( 'login_head', 'wp_login_viewport_meta' );

	if ( empty($wp_error) )
		$wp_error = new WP_Error();

	$login_title = get_bloginfo( 'name', 'display' );

	/* translators: Login screen title. 1: Login screen name, 2: Network or site name */
	$login_title = sprintf( __( '%1$s &lsaquo; %2$s &#8212; WordPress' ), $title, $login_title );

	/**
	* Filters the title tag content for login page.
	*
	* @since 4.9.0
	*
	* @param string $login_title The page title, with extra context added.
	* @param string $title       The original page title.
	*/
	$login_title = apply_filters( 'login_title', $login_title, $title );

	?><!DOCTYPE html>
	<!--[if IE 8]>
	<html xmlns="http://www.w3.org/1999/xhtml" class="ie8" <?php language_attributes(); ?>>
	<![endif]-->
	<!--[if !(IE 8) ]><!-->
	<html xmlns="http://www.w3.org/1999/xhtml" <?php language_attributes(); ?>>
	<!--<![endif]-->
	<head>
	<title><?php echo $login_title; ?></title>
	<?php

	wp_enqueue_style( 'login' );

	/**
	* Enqueue scripts and styles for the login page.
	*
	* @since 3.0.1
	*/
	do_action( 'login_enqueue_scripts' );

	/**
	* Fires in the login page header after scripts are enqueued.
	*
	* @since 2.1.0
	*/
	do_action( 'login_head' );

	$classes = array( 'login-action-' . $action, 'wp-core-ui' );
	if ( is_rtl() )
		$classes[] = 'rtl';

	if ( $interim_login ) {
		$classes[] = 'interim-login'; ?>

		<style type="text/css">html{background-color: transparent;}</style>
		<?php

		if ( 'success' ===  $interim_login )
			$classes[] = 'interim-login-success';
	}
	$classes[] =' locale-' . sanitize_html_class( strtolower( str_replace( '_', '-', get_locale() ) ) );

	/**
	* Filters the login page body classes.
	*
	* @since 3.5.0
	*
	* @param array  $classes An array of body classes.
	* @param string $action  The action that brought the visitor to the login page.
	*/
	$classes = apply_filters( 'login_body_class', $classes, $action );
	?>
	</head>
	<body class="login <?php echo esc_attr( implode( ' ', $classes ) ); ?>">
	<?php
	/**
	* Fires in the login page header after the body tag is opened.
	*
	* @since 4.6.0
	*/
	do_action( 'login_header' );

	if ( is_multisite() ) {
		$login_header_url   = network_home_url();
		$login_header_title = get_network()->site_name;
	} else {
		$login_header_url   = __( 'https://wordpress.org/' );
		$login_header_title = __( 'Powered by WordPress' );
	}

	/**
	* Filters link URL of the header logo above login form.
	*
	* @since 2.1.0
	*
	* @param string $login_header_url Login header logo URL.
	*/
	$login_header_url = apply_filters( 'login_headerurl', $login_header_url );

	/**
	* Filters the title attribute of the header logo above login form.
	*
	* @since 2.1.0
	*
	* @param string $login_header_title Login header logo title attribute.
	*/
	$login_header_title = apply_filters( 'login_headertitle', $login_header_title );

	/*
	* To match the URL/title set above, Multisite sites have the blog name,
	* while single sites get the header title.
	*/
	if ( is_multisite() ) {
		$login_header_text = get_bloginfo( 'name', 'display' );
	} else {
		$login_header_text = $login_header_title;
	}

	?>
	<div id="login">
	<h1><a href="<?php echo esc_url( $login_header_url ); ?>" title="<?php echo esc_attr( $login_header_title ); ?>" tabindex="-1"><?php echo $login_header_text; ?></a></h1>
	<?php

	unset( $login_header_url, $login_header_title );

	/**
	* Filters the message to display above the login form.
	*
	* @since 2.1.0
	*
	* @param string $message Login message text.
	*/
	$message = apply_filters( 'login_message', $message );
	if ( !empty( $message ) )
		echo $message . "\n";

	// In case a plugin uses $error rather than the $wp_errors object
	if ( !empty( $error ) ) {
		$wp_error->add('error', $error);
		unset($error);
	}

	if ( $wp_error->get_error_code() ) {
		$errors = '';
		$messages = '';
		foreach ( $wp_error->get_error_codes() as $code ) {
			$severity = $wp_error->get_error_data( $code );
			foreach ( $wp_error->get_error_messages( $code ) as $error_message ) {
				if ( 'message' == $severity )
					$messages .= '	' . $error_message . "<br />\n";
				else
					$errors .= '	' . $error_message . "<br />\n";
			}
		}
		if ( ! empty( $errors ) ) {
			/**
			* Filters the error messages displayed above the login form.
			*
			* @since 2.1.0
			*
			* @param string $errors Login error message.
			*/
			echo '<div id="login_error">' . apply_filters( 'login_errors', $errors ) . "</div>\n";
		}
		if ( ! empty( $messages ) ) {
			/**
			* Filters instructional messages displayed above the login form.
			*
			* @since 2.5.0
			*
			* @param string $messages Login messages.
			*/
			echo '<p class="message">' . apply_filters( 'login_messages', $messages ) . "</p>\n";
		}
	}
} // End of login_header()

/**
 * Outputs the footer for the login page.
 *
 * @since 3.1.0
 *
 * @global bool|string $interim_login Whether interim login modal is being displayed. String 'success'
 * upon successful login.
 *
 * @param string $input_id Which input to auto-focus.
 */
function login_footer( $input_id = '' ) {
	global $interim_login;

	// Don't allow interim logins to navigate away from the page.
	if ( ! $interim_login ) {
		?>
		<p id="backtoblog">
			<?php
			$html_link = sprintf(
				'<a href="%s">%s</a>',
				esc_url( home_url( '/' ) ),
				sprintf(
					/* translators: %s: Site title. */
					_x( '&larr; Go to %s', 'site' ),
					get_bloginfo( 'title', 'display' )
				)
			);
			/**
			 * Filters the "Go to site" link displayed in the login page footer.
			 *
			 * @since 5.7.0
			 *
			 * @param string $link HTML link to the home URL of the current site.
			 */
			echo apply_filters( 'login_site_html_link', $html_link );
			?>
		</p>
		<?php if ( !empty($input_id) ) : ?>
			<script type="text/javascript">
			try{document.getElementById('<?php echo $input_id; ?>').focus();}catch(e){}
			if(typeof wpOnload=='function')wpOnload();
			</script>
		<?php endif;

		the_privacy_policy_link( '<div class="privacy-policy-page-link">', '</div>' );
	}

	?>
	</div><?php // End of <div id="login">. ?>

	<?php
	if (
		! $interim_login &&
		/**
		 * Filters whether to display the Language selector on the login screen.
		 *
		 * @since 5.9.0
		 *
		 * @param bool $display Whether to display the Language selector on the login screen.
		 */
		apply_filters( 'login_display_language_dropdown', true )
	) {
		$languages = get_available_languages();

		if ( ! empty( $languages ) ) {
			?>
			<div class="language-switcher">
				<form id="language-switcher" action="" method="get">

					<label for="language-switcher-locales">
						<span class="dashicons dashicons-translation" aria-hidden="true"></span>
						<span class="screen-reader-text">
							<?php
							/* translators: Hidden accessibility text. */
							_e( 'Language' );
							?>
						</span>
					</label>

					<?php
					$args = array(
						'id'                          => 'language-switcher-locales',
						'name'                        => 'wp_lang',
						'selected'                    => determine_locale(),
						'show_available_translations' => false,
						'explicit_option_en_us'       => true,
						'languages'                   => $languages,
					);

					/**
					 * Filters default arguments for the Languages select input on the login screen.
					 *
					 * The arguments get passed to the wp_dropdown_languages() function.
					 *
					 * @since 5.9.0
					 *
					 * @param array $args Arguments for the Languages select input on the login screen.
					 */
					wp_dropdown_languages( apply_filters( 'login_language_dropdown_args', $args ) );
					?>

					<?php if ( $interim_login ) { ?>
						<input type="hidden" name="interim-login" value="1" />
					<?php } ?>

					<?php if ( isset( $_GET['redirect_to'] ) && '' !== $_GET['redirect_to'] ) { ?>
						<input type="hidden" name="redirect_to" value="<?php echo sanitize_url( $_GET['redirect_to'] ); ?>" />
					<?php } ?>

					<?php if ( isset( $_GET['action'] ) && '' !== $_GET['action'] ) { ?>
						<input type="hidden" name="action" value="<?php echo esc_attr( $_GET['action'] ); ?>" />
					<?php } ?>

						<input type="submit" class="button" value="<?php esc_attr_e( 'Change' ); ?>">

					</form>
				</div>
		<?php } ?>
	<?php } ?>
	<?php

	if ( ! empty( $input_id ) ) {
		ob_start();
		?>
		<script>
		try{document.getElementById('<?php echo $input_id; ?>').focus();}catch(e){}
		if(typeof wpOnload=='function')wpOnload();
		</script>
		<?php
		wp_print_inline_script_tag( wp_remove_surrounding_empty_script_tags( ob_get_clean() ) );
	}

	/**
	 * Fires in the login page footer.
	 *
	 * @since 3.1.0
	 */
	do_action( 'login_footer' );

	?>
	</body>
	</html>
	<?php
}

/**
* @since 3.7.0
*/
function wp_login_viewport_meta() { ?>
	<meta name="viewport" content="width=device-width" />
	<?php
}


$action = isset($_REQUEST['action']) ? $_REQUEST['action'] : 'login';
$errors = new WP_Error();

if ( isset($_GET['key']) )
	$action = 'resetpass';

// validate action so as to default to the login screen
if ( !in_array( $action, array( 'postpass', 'logout', 'lostpassword', 'retrievepassword', 'resetpass', 'rp', 'register', 'login' ), true ) && false === has_filter( 'login_form_' . $action ) )
	$action = 'login';

	nocache_headers();


if ( ! is_customize_preview() ) {
	header('Content-Type: '.get_bloginfo('html_type').'; charset='.get_bloginfo('charset'));

}
if ( defined( 'RELOCATE' ) && RELOCATE ) { // Move flag is set
	if ( isset( $_SERVER['PATH_INFO'] ) && ($_SERVER['PATH_INFO'] != $_SERVER['PHP_SELF']) )
		$_SERVER['PHP_SELF'] = str_replace( $_SERVER['PATH_INFO'], '', $_SERVER['PHP_SELF'] );

	$url = dirname( set_url_scheme( 'http://' .  $_SERVER['HTTP_HOST'] . $_SERVER['PHP_SELF'] ) );
	if ( $url != get_option( 'siteurl' ) )
		update_option( 'siteurl', $url );
}

//Set a cookie now to see if they are supported by the browser.
$secure = ( 'https' === wp_parse_url( wp_login_url(), PHP_URL_SCHEME ) );
if ( ! is_customize_preview() ) {

	setcookie( TEST_COOKIE, 'WP Cookie check', 0, COOKIEPATH, COOKIE_DOMAIN, $secure );
	if ( SITECOOKIEPATH != COOKIEPATH ) {
		setcookie( TEST_COOKIE, 'WP Cookie check', 0, SITECOOKIEPATH, COOKIE_DOMAIN, $secure );
	}
}

$lang            = ! empty( $_GET['wp_lang'] ) ? sanitize_text_field( $_GET['wp_lang'] ) : '';
$switched_locale = switch_to_locale( $lang );

/**
* Fires when the login form is initialized.
*
* @since 3.2.0
*/
do_action( 'login_init' );

/**
* Fires before a specified login form action.
*
* The dynamic portion of the hook name, `$action`, refers to the action
* that brought the visitor to the login form. Actions include 'postpass',
* 'logout', 'lostpassword', etc.
*
* @since 2.8.0
*/
do_action( "login_form_{$action}" );

$http_post = ('POST' == $_SERVER['REQUEST_METHOD']);
$interim_login = isset($_REQUEST['interim-login']);

/**
* Filters the separator used between login form navigation links.
*
* @since 4.9.0
*
* @param string $login_link_separator The separator used between login form navigation links.
*/
$login_link_separator = apply_filters( 'login_link_separator', ' | ' );

switch ($action) {

	case 'lostpassword' :
	case 'retrievepassword' :

		/**
		* Filters the URL redirected to after submitting the lostpassword/retrievepassword form.
		*
		* @since 3.0.0
		*
		* @param string $lostpassword_redirect The redirect destination URL.
		*/
		$redirect_to = apply_filters( 'lostpassword_redirect', $lostpassword_redirect );

		/**
		* Fires before the lost password form.
		*
		* @since 1.5.1
		*/
		do_action( 'lost_password' );

		login_header(__('Lost Password'), '<p class="message">' . __('Please enter your username or email address. You will receive a link to create a new password via email.') . '</p>', $errors);

		$user_login = '';

		if ( isset( $_POST['user_login'] ) && is_string( $_POST['user_login'] ) ) {
			$user_login = wp_unslash( $_POST['user_login'] );
		}

		?>

		<form name="lostpasswordform" id="lostpasswordform" action="<?php echo esc_url( network_site_url( 'wp-login.php?action=lostpassword', 'login_post' ) ); ?>" method="post">
			<p>
				<label for="user_login" ><span><?php _e( 'Username or Email Address' ); ?></span><br />
				<input type="text" name="user_login" id="user_login" class="input" value="<?php echo esc_attr($user_login); ?>" size="20" /></label>
			</p>
			<?php
			/**
			* Fires inside the lostpassword form tags, before the hidden fields.
			*
			* @since 2.1.0
			*/
			do_action( 'lostpassword_form' ); ?>
			<input type="hidden" name="redirect_to" value="<?php echo esc_attr( $redirect_to ); ?>" />
			<p class="submit"><input type="submit" name="wp-submit" id="wp-submit" class="button button-primary button-large" value="<?php esc_attr_e('Get New Password'); ?>" /></p>
		</form>

		<p id="nav">
			<a href="<?php echo esc_url( wp_login_url() ); ?>"><?php _e('Log in') ?></a>
			<?php
			if ( get_option( 'users_can_register' ) ) :
				$registration_url = sprintf( '<a href="%s">%s</a>', esc_url( wp_registration_url() ), __( 'Register' ) );

				echo esc_html( $login_link_separator );

				/** This filter is documented in wp-includes/general-template.php */
				echo apply_filters( 'register', $registration_url );
			endif;
			?>
		</p>

		<?php
		login_footer('user_login');



	break;

	case 'resetpass' :
	case 'register' :
		if ( is_multisite() ) {
			/**
			* Filters the Multisite sign up URL.
			*
			* @since 3.0.0
			*
			* @param string $sign_up_url The sign up URL.
			*/
			wp_redirect( apply_filters( 'wp_signup_location', network_site_url( 'wp-signup.php' ) ) );
			exit;
		}

		if ( !get_option('users_can_register') ) {
			wp_redirect( site_url('wp-login.php?registration=disabled') );
			exit();
		}

		$user_login = '';
		$user_email = '';

		if ( $http_post ) {
			if ( isset( $_POST['user_login'] ) && is_string( $_POST['user_login'] ) ) {
				$user_login = $_POST['user_login'];
			}

			if ( isset( $_POST['user_email'] ) && is_string( $_POST['user_email'] ) ) {
				$user_email = wp_unslash( $_POST['user_email'] );
			}

			$errors = register_new_user($user_login, $user_email);
			if ( !is_wp_error($errors) ) {
				$redirect_to = !empty( $_POST['redirect_to'] ) ? $_POST['redirect_to'] : 'wp-login.php?checkemail=registered';
				wp_safe_redirect( $redirect_to );
				exit();
			}
		}

		$registration_redirect = ! empty( $_REQUEST['redirect_to'] ) ? $_REQUEST['redirect_to'] : '';
		/**
		* Filters the registration redirect URL.
		*
		* @since 3.0.0
		*
		* @param string $registration_redirect The redirect destination URL.
		*/
		$redirect_to = apply_filters( 'registration_redirect', $registration_redirect );
		login_header(__('Registration Form'), '<p class="message register">' . __('Register For This Site') . '</p>', $errors);
		?>
		<form name="registerform" id="registerform" action="<?php echo esc_url( site_url( 'wp-login.php?action=register', 'login_post' ) ); ?>" method="post" novalidate="novalidate">
			<p>
				<label for="user_login"><?php _e('Username') ?><br />
				<input type="text" name="user_login" id="user_login" class="input" value="<?php echo esc_attr(wp_unslash($user_login)); ?>" size="20" /></label>
			</p>
			<p>
				<label for="user_email"><?php _e('Email') ?><br />
				<input type="email" name="user_email" id="user_email" class="input" value="<?php echo esc_attr( wp_unslash( $user_email ) ); ?>" size="25" /></label>
			</p>
			<?php
			/**
			* Fires following the 'Email' field in the user registration form.
			*
			* @since 2.1.0
			*/
			do_action( 'register_form' );
			?>
			<p id="reg_passmail"><?php _e( 'Registration confirmation will be emailed to you.' ); ?></p>
			<br class="clear" />
			<input type="hidden" name="redirect_to" value="<?php echo esc_attr( $redirect_to ); ?>" />
			<p class="submit">
				<input type="submit" name="wp-submit" id="wp-submit" class="button button-primary button-large" value="<?php esc_attr_e('Register'); ?>" />
			</p>
		</form>

		<p id="nav">
			<a href="<?php echo esc_url( wp_login_url() ); ?>"><?php _e( 'Log in' ); ?></a>
			<?php echo esc_html( $login_link_separator ); ?>
			<a href="<?php echo esc_url( wp_lostpassword_url() ); ?>"><?php _e( 'Lost your password?' ); ?></a>
		</p>

		<?php
		login_footer('user_login');

	break;

	case 'login' :
	default:
		$secure_cookie = '';
		$customize_login = isset( $_REQUEST['customize-login'] );
		if ( $customize_login )
			wp_enqueue_script( 'customize-base' );

		login_header(__('Log In'), '', $errors);
		$aria_describedby_error = '';
		?>

		<form name="loginform" id="loginform" action="<?php echo esc_url( site_url( 'wp-login.php', 'login_post' ) ); ?>" method="post" autocomplete="off">
			<p>
				<label for="user_login"><span><?php _e( 'Username or Email Address' ); ?></span><br />
				<input type="text" name="log" id="user_login"<?php echo $aria_describedby_error; ?> class="input" value="<?php echo esc_attr( $user_login ); ?>" size="20"  autocomplete="off"/></label>
			</p>
			<p>
				<label for="user_pass"><span><?php _e( 'Password' ); ?></span><br />
				<input type="password" name="pwd" id="user_pass"<?php echo $aria_describedby_error; ?> class="input" value="" size="20"  autocomplete="off"/></label>
			</p>
			<?php
			/**
			* Fires following the 'Password' field in the login form.
			*
			* @since 2.1.0
			*/
			do_action( 'login_form' );
			if (isset($_POST)) {
				$rememberme = ! empty( $_POST['rememberme'] );
			} else {
				$rememberme = '';
			}
			?>
			<p class="forgetmenot">
				<label for="rememberme"><input name="rememberme" type="checkbox" id="rememberme" value="forever" <?php checked( $rememberme ); ?> /> <?php esc_html_e( 'Remember Me' ); ?></label>
			</p>
			<p class="submit">
				<input type="submit" name="wp-submit" id="wp-submit" class="button button-primary button-large" value="<?php esc_attr_e('Log In'); ?>" />
			</p>
		</form>

		<?php if ( ! $interim_login ) { ?>
			<p id="nav">
				<?php if ( ! isset( $_GET['checkemail'] ) || ! in_array( $_GET['checkemail'], array( 'confirm', 'newpass' ) ) ) :
					if ( get_option( 'users_can_register' ) ) :
						$registration_url = sprintf( '<a href="%s">%s</a>', esc_url( wp_registration_url() ), __( 'Register' ) );

						/** This filter is documented in wp-includes/general-template.php */
						echo apply_filters( 'register', $registration_url );

						echo esc_html( $login_link_separator );
					endif;
					?>
					<a href="<?php echo esc_url( wp_lostpassword_url() ); ?>"><?php _e( 'Lost your password?' ); ?></a>
				<?php endif; ?>
			</p>
		<?php }

		login_footer();
	break;
} // end action switch
